<?php
namespace App\Inventory;

use App\Core\Database;

class StockService
{
    /* =============================
       REAL AVAILABLE STOCK
       quantity - reserved
    ============================== */
    public function getAvailableStock(int $productId, int $branchId): int
    {
        $row = Database::fetchOne(
            "SELECT quantity 
             FROM product_stocks 
             WHERE product_id = ? 
             AND branch_id = ?",
            [$productId, $branchId]
        );

        $reserved = Database::fetchOne(
            "SELECT COALESCE(SUM(quantity),0) as total
             FROM stock_movements
             WHERE product_id = ?
             AND branch_id = ?
             AND type = 'reserve'
             AND reference_type = 'suspend'
             AND reversed = 0",
            [$productId, $branchId]
        );

        $physical = (int)($row['quantity'] ?? 0);
        $reservedQty = (int)($reserved['total'] ?? 0);

        return $physical - $reservedQty;
    }

    /* =============================
       VALIDATE BEFORE ADD TO CART
    ============================== */
    public function validateStock(int $productId, int $branchId, int $qty): bool
    {
        return $this->getAvailableStock($productId, $branchId) >= $qty;
    }

    /* =============================
       RESERVE STOCK (ON SUSPEND)
    ============================== */
public function reserveStock(
    int $productId,
    int $branchId,
    int $qty,
    int $suspendId
)
{
    // 🔒 1️⃣ Lock physical stock row
    $row = Database::fetchOne(
        "SELECT quantity
         FROM product_stocks
         WHERE product_id = ?
         AND branch_id = ?
         FOR UPDATE",
        [$productId, $branchId]
    );

    if (!$row) {
        throw new \Exception("Stock record missing for product {$productId}");
    }

    $physical = (int)$row['quantity'];

    // 🔒 2️⃣ Calculate reserved while locked
    $reserved = Database::fetchOne(
        "SELECT COALESCE(SUM(quantity),0) as total
         FROM stock_movements
         WHERE product_id = ?
         AND branch_id = ?
         AND type = 'reserve'
         AND reference_type = 'suspend'
         AND reversed = 0
         FOR UPDATE",
        [$productId, $branchId]
    );

    $reservedQty = (int)$reserved['total'];
    $available   = $physical - $reservedQty;

    // 🚫 3️⃣ Hard validation
    if ($available < $qty) {
        throw new \Exception(
            "Cannot reserve stock. Not enough available for product {$productId}"
        );
    }

    // ✅ 4️⃣ Record reserve movement
    $this->recordMovement(
        $productId,
        $branchId,
        'reserve',
        $qty,
        'suspend',
        $suspendId
    );
}


    /* =============================
       RELEASE RESERVED STOCK
       (SAFE VERSION)
    ============================== */
    public function releaseReserve(
        int $productId,
        int $branchId,
        int $qty,
        int $suspendId
    )
    {
        // Mark ONLY this product's reserve as reversed
        Database::execute(
            "UPDATE stock_movements
             SET reversed = 1
             WHERE product_id = ?
             AND branch_id = ?
             AND reference_id = ?
             AND reference_type = 'suspend'
             AND type = 'reserve'
             AND reversed = 0",
            [
                $productId,
                $branchId,
                $suspendId
            ]
        );

        // Optional: log release action
        $this->recordMovement(
            $productId,
            $branchId,
            'release',
            $qty,
            'suspend',
            $suspendId
        );
    }

    /* =============================
       FINAL DEDUCT (ON CHECKOUT)
    ============================== */
    public function deductStock(
        int $productId,
        int $branchId,
        int $qty,
        int $saleId
    )
    {
        // Safety: never allow negative stock
        $row = Database::fetchOne(
            "SELECT quantity
             FROM product_stocks
             WHERE product_id = ?
             AND branch_id = ?
             FOR UPDATE",
            [$productId, $branchId]
        );

        if (!$row) {
            throw new \Exception("Stock record missing for product {$productId}");
        }

        $current = (int)$row['quantity'];

        if ($current < $qty) {
            throw new \Exception("Insufficient physical stock for product {$productId}");
        }

        Database::execute(
            "UPDATE product_stocks
             SET quantity = quantity - ?
             WHERE product_id = ?
             AND branch_id = ?",
            [$qty, $productId, $branchId]
        );

        $this->recordMovement(
            $productId,
            $branchId,
            'sale',
            $qty,
            'sale',
            $saleId
        );
    }

    /* =============================
       REVERSE SUSPEND RESERVE
    ============================== */
    
public function reverseSuspendReserve(
    int $productId,
    int $branchId,
    int $suspendId
) {

    // Lock reserve rows first
    $reserved = Database::fetchOne(
        "SELECT COALESCE(SUM(quantity),0) as total
         FROM stock_movements
         WHERE product_id = ?
         AND branch_id = ?
         AND type = 'reserve'
         AND reference_type = 'suspend'
         AND reference_id = ?
         AND reversed = 0
         FOR UPDATE",
        [$productId, $branchId, $suspendId]
    );

    $reservedQty = (int)$reserved['total'];

    if ($reservedQty <= 0) {
        return;
    }

    // Reverse reserve safely
    Database::execute(
        "UPDATE stock_movements
         SET reversed = 1
         WHERE product_id = ?
         AND branch_id = ?
         AND type = 'reserve'
         AND reference_type = 'suspend'
         AND reference_id = ?
         AND reversed = 0",
        [$productId, $branchId, $suspendId]
    );

    // Log release movement (full audit trail)
    $this->recordMovement(
        $productId,
        $branchId,
        'release',
        $reservedQty,
        'suspend',
        $suspendId
    );
}

    /* =============================
       STOCK MOVEMENT LOGGER
    ============================== */
    private function recordMovement(
        int $productId,
        int $branchId,
        string $type,
        int $qty,
        string $referenceType,
        int $referenceId
    )
    {
        Database::execute(
            "INSERT INTO stock_movements
            (product_id, branch_id, type, quantity, reference_type, reference_id, reversed, created_at)
            VALUES (?, ?, ?, ?, ?, ?, 0, NOW())",
            [
                $productId,
                $branchId,
                $type,
                $qty,
                $referenceType,
                $referenceId
            ]
        );
    }
}
