<?php
namespace App\POS\Controllers;

use App\Core\Database;
use App\Core\Auth;
use App\POS\PosService;
use App\Inventory\StockService;
use App\POS\PosAuditService;

class PosController
{
public function __construct()
{
    // Must be logged in
    if (!Auth::check()) {
        header('Location: /login');
        exit;
    }

    // Must have POS permission
    if (!Auth::hasPermission('pos_access')){
        http_response_code(403);
require __DIR__ . '/../../../resources/views/errors/403.php';
        exit;
    }
}

    public function index()
{
        if (!isset($_SESSION['pos_cart'])) {
            $_SESSION['pos_cart'] = ['items'=>[], 'total'=>0];
        }
        
        $_SESSION['checkout_token'] = bin2hex(random_bytes(16));

        /* ================= AUTO ADD NEW PRODUCT FROM CREATE ================= */

if (isset($_GET['add_product_id'])) {

    $productId = (int)$_GET['add_product_id'];
    $branchId  = $_SESSION['branch_id'];

    $product = Database::fetchOne(
        "SELECT id, name, sale_price 
         FROM products 
         WHERE id = ?",
        [$productId]
    );

    if ($product) {

        $found = false;

        foreach ($_SESSION['pos_cart']['items'] as &$item) {
            if ($item['id'] == $product['id']) {
                $item['qty']++;
                $found = true;
                break;
            }
        }

        if (!$found) {
            $_SESSION['pos_cart']['items'][] = [
                'id'    => $product['id'],
                'name'  => $product['name'],
                'price' => (float)$product['sale_price'],
                'qty'   => 1
            ];
        }

        $this->recalc();
    }
}

        $branchId = $_SESSION['branch_id'];
        $this->recalc();
        $cart = $_SESSION['pos_cart'];
        $selectedCustomer = $_SESSION['pos_customer'] ?? null;

        $employees = Database::fetchAll(
            "SELECT id, name 
             FROM employees 
             WHERE branch_id = ? 
             ORDER BY name ASC",
            [$branchId]
        );

        if (!isset($_SESSION['pos_salesperson'])) {
            $_SESSION['pos_salesperson'] = $_SESSION['employee']['id'];
        }

        $currentSalespersonId = $_SESSION['pos_salesperson'];

        ob_start();
        require __DIR__ . '/../../../resources/views/pos/index.php';
        $content = ob_get_clean();

        require __DIR__ . '/../../../resources/views/layouts/layout.php';
    }

    /* ================= CHANGE SALESPERSON ================= */

    public function changeSalesperson()
    {
        
        header('Content-Type: application/json');
        
        if (!Auth::hasPermission('pos_change_salesperson')) {
    echo json_encode(['success'=>false]);
    return;
}

        $data = json_decode(file_get_contents('php://input'), true);
        $employeeId = (int)($data['id'] ?? 0);
        $branchId = $_SESSION['branch_id'];

        $employee = Database::fetchOne(
            "SELECT id FROM employees WHERE id = ? AND branch_id = ?",
            [$employeeId, $branchId]
        );

        if (!$employee) {
            echo json_encode(['success'=>false]);
            return;
        }

        $_SESSION['pos_salesperson'] = $employeeId;

        echo json_encode(['success'=>true]);
    }

    /* ================= Discount Entire Sale % ================= */
public function setDiscount()
{
    header('Content-Type: application/json');

    if (!Auth::hasPermission('pos_discount')) {
        echo json_encode(['success'=>false]);
        return;
    }

    $data = json_decode(file_get_contents('php://input'), true);
    $percent = (float)($data['percent'] ?? 0);

    if (!Auth::hasPermission('pos_high_discount') && $percent > 20) {
        echo json_encode([
            'success'=>false,
            'message'=>'Discount above 20% requires manager approval.'
        ]);
        return;
    }

    $_SESSION['pos_discount'] = $percent;

    echo json_encode(['success'=>true]);
}

    /* ================= DETACH CUSTOMER ================= */
public function removeCustomer()
{
    unset($_SESSION['pos_customer']);
    echo json_encode(['success'=>true]);
}

/* ================= ENTERPRISE SUSPEND ================= */
public function suspendSale()

{
        header('Content-Type: application/json');
        
    if (!Auth::hasPermission('pos_suspend')) {
    echo json_encode(['success'=>false, 'message'=>'Permission denied']);
    return;
}

    if (empty($_SESSION['pos_cart']['items'])) {
        echo json_encode(['success'=>false]);
        return;
    }

    $data = json_decode(file_get_contents('php://input'), true);
    $note = trim($data['note'] ?? '');

    $cart = $_SESSION['pos_cart'];
    $branchId = $_SESSION['branch_id'];

    Database::begin();

    try {

        /* ===============================
           1️⃣ LOCK ALL STOCK ROWS FIRST
        =============================== */
        foreach ($cart['items'] as $item) {

            if ($item['id'] <= 0) continue;

            $lockRow = Database::fetchOne(
    "SELECT quantity
     FROM product_stocks
     WHERE product_id = ?
     AND branch_id = ?
     FOR UPDATE",
    [$item['id'], $branchId]
);

if (!$lockRow) {
    throw new \Exception('Stock record missing for product ID ' . $item['id']);
}

        }

        /* ===============================
           2️⃣ VALIDATE FULL CART
        =============================== */
        $stockService = new StockService();

        foreach ($cart['items'] as $item) {

            if ($item['id'] <= 0) continue;

            if (!$stockService->validateStock(
                $item['id'],
                $branchId,
                $item['qty']
            )) {
                throw new \Exception(
                    'Stock not enough for product ID ' . $item['id']
                );
            }
        }

        /* ===============================
           3️⃣ CREATE SUSPEND RECORD
        =============================== */
        $countToday = Database::fetchOne(
            "SELECT COUNT(*) AS total 
             FROM pos_suspended_sales 
             WHERE DATE(created_at) = CURDATE()
             AND branch_id = ?",
            [$branchId]
        );

        $sequence = str_pad(($countToday['total'] + 1), 4, '0', STR_PAD_LEFT);
        $reference = 'SUS-' . date('Ymd') . '-' . $sequence;

        Database::execute(
            "INSERT INTO pos_suspended_sales
            (reference_no, branch_id, customer_id,
             salesperson_id, cart_data, total,
             note, status, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, 'active', NOW())",
            [
                $reference,
                $branchId,
                $_SESSION['pos_customer']['id'] ?? null,
                $_SESSION['pos_salesperson'] ?? null,
                json_encode($cart),
                $cart['total'],
                $note
            ]
        );

        $suspendId = Database::lastInsertId();

        /* ===============================
           4️⃣ RESERVE STOCK
        =============================== */
        foreach ($cart['items'] as $item) {

            if ($item['id'] <= 0) continue;

            $stockService->reserveStock(
                $item['id'],
                $branchId,
                $item['qty'],
                $suspendId
            );
        }

        Database::commit();

$audit = new PosAuditService();
$audit->log(
    'suspend_created',
    $suspendId,
    'suspend',
    'Sale suspended successfully'
);

        unset($_SESSION['pos_cart']);
        unset($_SESSION['pos_customer']);
        unset($_SESSION['pos_salesperson']);

        echo json_encode([
            'success'=>true,
            'reference'=>$reference
        ]);

} catch (\Throwable $e) {

    Database::rollback();

    $audit = new PosAuditService();
    $audit->log(
        'suspend_failed',
        null,
        'suspend',
        $e->getMessage()
    );

    echo json_encode([
        'success'=>false,
        'message'=>$e->getMessage()
    ]);
}

}

/* ================= LIST SUSPEND ================= */

public function listSuspended()
{
    header('Content-Type: application/json');

    $branchId = $_SESSION['branch_id'];

    $rows = Database::fetchAll(
        "SELECT s.id, s.reference_no, s.total, s.note, s.created_at,
                c.name AS customer_name,
                e.name AS salesperson_name
         FROM pos_suspended_sales s
         LEFT JOIN customers c ON s.customer_id = c.id
         LEFT JOIN employees e ON s.salesperson_id = e.id
         WHERE s.branch_id = ? AND s.status = 'active'
         ORDER BY s.created_at DESC",
        [$branchId]
    );

    echo json_encode($rows);
}

/* ================= RESTORE SUSPEND ================= */

public function restoreSuspended()
{
        header('Content-Type: application/json');
        
    if (!Auth::hasPermission('pos_restore_suspend')) {
    echo json_encode(['success'=>false, 'message'=>'Permission denied']);
    return;
}

    $data = json_decode(file_get_contents('php://input'), true);
    $id = (int)($data['id'] ?? 0);

    if (!$id) {
        echo json_encode(['success'=>false]);
        return;
    }

    $row = Database::fetchOne(
        "SELECT * FROM pos_suspended_sales 
         WHERE id = ? AND branch_id = ? AND status = 'active'
",
        [$id, $_SESSION['branch_id']]

    );

    if (!$row) {
        echo json_encode(['success'=>false]);
        return;
    }

    $cart = json_decode($row['cart_data'], true);

    $_SESSION['pos_cart'] = $cart;
    $_SESSION['pos_discount'] = $cart['discount'] ?? 0;

    if ($row['customer_id']) {
        $customer = Database::fetchOne(
            "SELECT id, name, phone FROM customers WHERE id = ?",
            [$row['customer_id']]
        );
        $_SESSION['pos_customer'] = $customer;
    }

    if ($row['salesperson_id']) {
        $_SESSION['pos_salesperson'] = $row['salesperson_id'];
    }

    Database::execute(
        "UPDATE pos_suspended_sales 
         SET status = 'restored', restored_at = NOW() 
         WHERE id = ?",
        [$id]
    );

$_SESSION['restored_suspend_id'] = $id;

$audit = new PosAuditService();
$audit->log(
    'suspend_restored',
    $id,
    'suspend',
    'Suspended sale restored'
);

    echo json_encode([
        'success'=>true,
        'cart'=>$cart
    ]);
    
}


/* ================= CANCEL SUSPEND ================= */

public function cancelSuspended()
{
        header('Content-Type: application/json');
        
    if (!Auth::hasPermission('pos_cancel_suspend')) {
    echo json_encode(['success'=>false, 'message'=>'Permission denied']);
    return;
}

    $data = json_decode(file_get_contents('php://input'), true);
    $id = (int)($data['id'] ?? 0);
    $branchId = $_SESSION['branch_id'];

    if (!$id) {
        echo json_encode(['success'=>false]);
        return;
    }

    $row = Database::fetchOne(
        "SELECT * FROM pos_suspended_sales 
         WHERE id = ? AND branch_id = ? AND status = 'active'
",
        [$id, $branchId]
    );

    if (!$row) {
        echo json_encode(['success'=>false]);
        return;
    }

    $items = json_decode($row['cart_data'], true);

    $stockService = new StockService();

    foreach ($items['items'] as $item) {
        if ($item['id'] > 0) {
            $stockService->releaseReserve(
                $item['id'],
                $row['branch_id'],
                $item['qty'],
                $id
            );
        }
    }

    Database::execute(
        "UPDATE pos_suspended_sales 
         SET status = 'cancelled' 
         WHERE id = ?",
        [$id]
    );

$audit = new PosAuditService();
$audit->log(
    'suspend_cancelled',
    $id,
    'suspend',
    'Suspended sale cancelled'
);

    echo json_encode(['success'=>true]);
}


    /* ================= QUICK ADD ITEM ================= */
public function quickAddAjax()
{
    header('Content-Type: application/json');

    $data = json_decode(file_get_contents('php://input'), true);

    $name  = trim($data['name'] ?? '');
    $price = (float)($data['price'] ?? 0);

    if (!$name || $price <= 0) {
        echo json_encode(['success'=>false]);
        return;
    }

    $_SESSION['pos_cart']['items'][] = [
        'id'    => 0,
        'name'  => $name,
        'price' => $price,
        'qty'   => 1
    ];

    $this->recalc();

    echo json_encode($_SESSION['pos_cart']);
}

    /* ================= SET MODE ================= */
    
public function setMode()
{
    header('Content-Type: application/json');

    $data = json_decode(file_get_contents('php://input'), true);
    $_SESSION['pos_mode'] = $data['mode'] ?? 'sale';

    echo json_encode(['success'=>true]);
}

    /* ================= CANCEL SALE BUTTON ================= */
public function cancelSale()
{
    unset($_SESSION['pos_cart']);
    unset($_SESSION['pos_customer']);
    unset($_SESSION['pos_salesperson']);

    echo json_encode(['success'=>true]);
}

    /* ================= PRODUCT PICKER ================= */
public function productsGrid()
{
    $products = Database::fetchAll(
        "SELECT id, name, sale_price FROM products LIMIT 100"
    );

    header('Content-Type: application/json');
    echo json_encode($products);
}

    /* ================= SCAN ================= */

public function scanAjax()
{
    header('Content-Type: application/json');

    $data = json_decode(file_get_contents('php://input'), true);
    $barcode = $data['barcode'] ?? null;

    if (!$barcode) {
        echo json_encode($_SESSION['pos_cart']);
        return;
    }

    $product = Database::fetchOne(
        "SELECT id, name, sale_price 
         FROM products 
         WHERE barcode = ?",
        [$barcode]
    );

    if (!$product) {
        echo json_encode($_SESSION['pos_cart']);
        return;
    }

$stockService = new StockService();
$branchId = $_SESSION['branch_id'];

$currentQty = 0;

foreach ($_SESSION['pos_cart']['items'] as $item) {
    if ($item['id'] == $product['id']) {
        $currentQty = $item['qty'];
        break;
    }
}

$requestedQty = $currentQty + 1;

if (!$stockService->validateStock($product['id'], $branchId, $requestedQty)) {
    echo json_encode([
        'error' => 'Stock not enough. Available: ' . 
                   $stockService->getAvailableStock($product['id'], $branchId)
    ]);
    return;
}

    $found = false;

    foreach ($_SESSION['pos_cart']['items'] as &$item) {
        if ($item['id'] == $product['id']) {
            $item['qty']++;
            $found = true;
            break;
        }
    }

    if (!$found) {
        $_SESSION['pos_cart']['items'][] = [
            'id'    => $product['id'],
            'name'  => $product['name'],
            'price' => (float)$product['sale_price'],
            'qty'   => 1
        ];
    }

    $this->recalc();
    echo json_encode($_SESSION['pos_cart']);
}

    /* ================= UPDATE QTY +- BUTTON ================= */
    
    public function updateQtyAjax()
    {
        header('Content-Type: application/json');

        $data = json_decode(file_get_contents('php://input'), true);
        $i = (int)($data['index'] ?? -1);
        $action = $data['action'] ?? '';

        if (!isset($_SESSION['pos_cart']['items'][$i])) {
            echo json_encode($_SESSION['pos_cart']);
            return;
        }

if ($action === 'inc') {

    $productId = $_SESSION['pos_cart']['items'][$i]['id'];
    $branchId  = $_SESSION['branch_id'];
    $newQty    = $_SESSION['pos_cart']['items'][$i]['qty'] + 1;

    if ($productId > 0) {

        $stockService = new StockService();

        if (!$stockService->validateStock($productId, $branchId, $newQty)) {
            echo json_encode([
                'error' => 'Stock not enough. Available: ' .
                           $stockService->getAvailableStock($productId, $branchId)
            ]);
            return;
        }
    }

    $_SESSION['pos_cart']['items'][$i]['qty'] = $newQty;
}

        if ($action === 'dec') {
            $_SESSION['pos_cart']['items'][$i]['qty']--;
            if ($_SESSION['pos_cart']['items'][$i]['qty'] <= 0) {
                unset($_SESSION['pos_cart']['items'][$i]);
                $_SESSION['pos_cart']['items'] = array_values($_SESSION['pos_cart']['items']);
            }
        }

        $this->recalc();
        echo json_encode($_SESSION['pos_cart']);
    }

    public function removeItemAjax()
    {
        header('Content-Type: application/json');

        $data = json_decode(file_get_contents('php://input'), true);
        $i = (int)($data['index'] ?? -1);

        if (isset($_SESSION['pos_cart']['items'][$i])) {
            unset($_SESSION['pos_cart']['items'][$i]);
            $_SESSION['pos_cart']['items'] = array_values($_SESSION['pos_cart']['items']);
        }

        $this->recalc();
        echo json_encode($_SESSION['pos_cart']);
    }

private function recalc()
{
    $total = 0;
    foreach ($_SESSION['pos_cart']['items'] as $item) {
        $total += $item['price'] * $item['qty'];
    }

    $discount = $_SESSION['pos_discount'] ?? 0;

    if ($discount > 0) {
        $total = $total - ($total * ($discount/100));
    }

    $_SESSION['pos_cart']['total'] = $total;
}


    /* ================= CUSTOMER ================= */

    public function selectCustomer()
    {
        header('Content-Type: application/json');

        $data = json_decode(file_get_contents('php://input'), true);
        $id = (int)($data['id'] ?? 0);

        if ($id === 0) {
            unset($_SESSION['pos_customer']);
            echo json_encode(['success'=>true]);
            return;
        }

        $customer = Database::fetchOne(
            "SELECT id, name, phone FROM customers WHERE id = ?",
            [$id]
        );

        if ($customer) {
            $_SESSION['pos_customer'] = $customer;
            echo json_encode(['success'=>true]);
            return;
        }

        echo json_encode(['success'=>false]);
    }

    /* ================= CHECKOUT ================= */
public function checkout()
{
    header('Content-Type: application/json');
$data = json_decode(file_get_contents('php://input'), true);
if (($data['token'] ?? '') !== ($_SESSION['checkout_token'] ?? '')) {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid or duplicate checkout request.'
    ]);
    return;
}

    unset($_SESSION['checkout_token']);

if (!Auth::hasPermission('pos_checkout')) {
    echo json_encode([
        'success' => false,
        'message' => 'Permission denied.'
    ]);
    return;
}

    if (empty($_SESSION['pos_cart']['items'])) {
        echo json_encode([
            'success' => false,
            'message' => 'Cart is empty'
        ]);
        return;
    }

    $paymentMethod = $data['payment_method'] ?? null;
    $tendered = isset($data['amount_tendered']) ? (float)$data['amount_tendered'] : null;
    $transactionUUID = $data['transaction_uuid'] ?? null;

if (!$transactionUUID) {
    echo json_encode([
        'success' => false,
        'message' => 'Transaction ID missing'
    ]);
    return;
}

    $total = (float)$_SESSION['pos_cart']['total'];

    if (!$paymentMethod) {
        echo json_encode([
            'success' => false,
            'message' => 'Payment method required'
        ]);
        return;
    }

    $changeAmount = 0;

    if ($paymentMethod === 'cash' && $tendered !== null) {
        $changeAmount = max(0, $tendered - $total);
    }

    $service = new PosService();

    $result = $service->checkout(
    $_SESSION['pos_cart'],
    $_SESSION['pos_customer'] ?? null,
    $paymentMethod,
    $tendered,
    $changeAmount,
    $transactionUUID
);


    if (!$result['success']) {
        echo json_encode($result);
        return;
    }

    unset($_SESSION['pos_cart']);
    unset($_SESSION['pos_customer']);
    unset($_SESSION['pos_salesperson']);
    unset($_SESSION['restored_suspend_id']);

    echo json_encode($result);
}

}

